<?php
/*--------------------------------------------------------------
   GambioRevenueTrend.php 2021-07-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\TimespanDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use Gambio\Core\Application\ValueObjects\UserPreferences;

class GambioRevenueTrend extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.revenue.trend';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,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";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Umsatz (Trend)',
        self::LANGUAGE_CODE_ENGLISH => 'Turnover (Trend)'
    ];
    
    private const CATEGORY_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Umsatz',
        self::LANGUAGE_CODE_ENGLISH => 'Revenue'
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    /**
     * @var Connection
     */
    private $connection;
    
    
    /**
     * @inheritDoc
     */
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory $dataFactory,
        UserPreferences $userPreferences,
        Connection $connection
    ) {
        $this->dataFactory     = $dataFactory;
        $this->userPreferences = $userPreferences;
        $this->connection      = $connection;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createAreaChart(),
                            $overviewFactory->useOptions()->createOptions($overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createTimespanDropdown($overviewFactory)),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    /**
     * @inheritDoc
     */
    public function data(Options $options): Data
    {
        $categories    = [];
        $values        = [];
        $categoryTitle = self::CATEGORY_TITLE[$this->userPreferences->languageId()
                                              === self::LANGUAGE_ID_GERMAN ? self::LANGUAGE_CODE_GERMAN : self::LANGUAGE_CODE_ENGLISH];
        $timespan      = $this->dataFactory->useTimespan()->createFromTerm($options->getById(TimespanDropdown::ID)
                                                                               ->value());
        $parameters    = [
            ':excludedOrderIds' => implode(',', self::EXCLUDED_ORDER_STATUS_IDS),
            ':startDate'        => $timespan->startDate()->format(self::QUERY_TIMESPAN_FORMAT_DATE_START),
            ':endDate'          => $timespan->endDate()->format(self::QUERY_TIMESPAN_FORMAT_DATE_END)
        ];
        $revenues      = $this->connection->createQueryBuilder()
            ->select([
                         'SUM(ROUND(orders_total.value / orders.currency_value, 2)) AS sales',
                         'UNIX_TIMESTAMP(DATE(orders.date_purchased)) AS date'
                     ])
            ->from('orders')
            ->join('orders',
                   'orders_total',
                   'orders_total',
                   'orders.orders_id = orders_total.orders_id')
            ->where('orders_status NOT IN (:excludedOrderIds)')
            ->andWhere('orders_total.class="ot_total"')
            ->andWhere('orders.date_purchased BETWEEN :startDate AND :endDate')
            ->groupBy('date')
            ->orderBy('date')
            ->setParameters($parameters)
            ->execute()
            ->fetchAll();
        
        $taxesRaw = $this->connection->createQueryBuilder()
            ->select([
                         'UNIX_TIMESTAMP(DATE(orders.date_purchased)) AS purchased_date',
                         'SUM(ROUND(orders_total.value / orders.currency_value, 2)) AS taxes'
                     ])
            ->from('orders')
            ->join('orders',
                   'orders_total',
                   'orders_total',
                   'orders.orders_id = orders_total.orders_id')
            ->where('orders_total.class="ot_tax"')
            ->andWhere('orders_status NOT IN (:excludedOrderIds)')
            ->andWhere('orders.date_purchased BETWEEN :startDate AND :endDate')
            ->setParameters($parameters)
            ->groupBy('purchased_date')
            ->orderBy('purchased_date')
            ->execute()
            ->fetchAll();
        
        $taxes = [];
        
        foreach ($taxesRaw as ['purchased_date' => $date, 'taxes' => $tax]) {
            $taxes[$date] = (float)$tax;
        }
        
        foreach ($revenues as ['date' => $date, 'sales' => $sales]) {
            $categories[] = $this->dataFactory->useSerialData()->createCategory((string)$date);
            $values[]     = $this->dataFactory->useSerialData()->createItemValue(isset($taxes[$date]) ? (float)$sales
                                                                                                        - (float)$taxes[$date] : (float)$sales);
        }
        
        return $this->dataFactory->useSerialData()->createSerialData($this->dataFactory->useSerialData()
                                                                         ->createCategories(...$categories),
                                                                     $this->dataFactory->useSerialData()
                                                                         ->createItems($this->dataFactory->useSerialData()
                                                                                           ->createItem($this->dataFactory->useSerialData()
                                                                                                            ->createItemTitle($categoryTitle),
                                                                                                        $this->dataFactory->useSerialData()
                                                                                                            ->createItemValues(...
                                                                                                                $values))));
    }
}
